---------------------------------------------------------------------------
-- This VHDL file was developed by Daniel Llamocca (2014).  It may be
-- freely copied and/or distributed at no cost.  Any persons using this
-- file for any purpose do so at their own risk, and are responsible for
-- the results of such use.  Daniel Llamocca does not guarantee that
-- this file is complete, correct, or fit for any particular purpose.
-- NO WARRANTY OF ANY KIND IS EXPRESSED OR IMPLIED.  This notice must
-- accompany any copy of this file.
--------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use ieee.math_real.log2;
use ieee.math_real.ceil;

library UNISIM;
use UNISIM.vcomponents.all;

library work;
use work.pack_xtras.all;
--use work.lpm_components.all;

-- NPIXELS: Only square images allowed, with NPIXELS being a power of 2.
entity vga_ctrl_ram is	
	generic (clock_pixel_ratio : integer:= 2;
	         NPIXELS: INTEGER:= 256; -- Picture: NPIXELSxNPIXELS. Max for Nexys-4: 525x525
				                        -- So far, we only work with square images
				FILE_IMG: STRING:= "myimg.txt"; -- text file containing the image
	         nbits: integer:= 12);  -- number of bits for each pixel. Example: 3, 8, 12, 15 
	port ( clock: in std_logic;	
			 resetn: in std_logic;	   
			 SW: in std_logic_vector (nbits-1 downto 0);
			 RGB: out std_logic_vector (nbits-1 downto 0);
			 HS, VS: out std_logic;
			 vga_clk: out std_logic;

			 -- debug signals
			 video_on: out std_logic;
			 hcount, vcount: out std_logic_vector (9 downto 0));
end vga_ctrl_ram;

architecture structure of vga_ctrl_ram is

    component in_RAMgen
        generic ( nrows: integer:= 128;
                  ncols: integer:= 128;
    --	          NDATA : integer:= 128*128); -- 256*256, 128*128 -- DATA samples, each of B bits (B<=16)
                 -- so far for 2D memories, let's only accept PxP, where P is a power of 2
                 FILE_IMG: string:="myinival.txt"; -- initial values for the memory: (256*256)x16 bits. This is fixed (for now).
                 INIT_VALUES: string:= "NO"); -- "YES" "NO". If "NO", FILE_IMG is not considered.
        port (   clock: in std_logic;	
                 inRAM_idata: in std_logic_vector (15 downto 0); -- up to 16 bits
                 inRAM_add: in std_logic_vector (integer(ceil(log2(real(nrows*ncols) ) ))-1 downto 0);
                 inRAM_we, inRAM_en: in std_logic;
                 inRAM_odata: out std_logic_vector (15 downto 0)); -- 16 bits output			 
    end component;
	
	constant NDATA: INTEGER:= NPIXELS*NPIXELS;
	constant NBPR: INTEGER:= ceil_log2(NPIXELS); -- bits per row (or per column)
	-- NPIXELS = 256 -> NBPR = 8 -> we only get the 8 LSBs of vcount_buf and hcount_buf
	signal inRAM_idata: std_logic_vector (15 downto 0);
	signal inRAM_add: std_logic_vector (ceil_log2(NDATA)-1 downto 0);
	signal inRAM_we, inRAM_en: std_logic;
	signal inRAM_odata:  std_logic_vector (15 downto 0); 
	
	signal RGB_buf: std_logic_vector (nbits-1 downto 0);
	signal vga_tick: std_logic;
	signal video_on_buf: std_logic;
	signal hcount_buf, vcount_buf: std_logic_vector (9 downto 0);

	signal in_RGB: std_logic_vector (nbits-1 downto 0);
	signal sel_RGB: std_logic;
	
begin

a1: assert (NPIXELS <= 480)
    report "We only accept up to 480 (Nexys-4 Board)!" -- 525 is the limit, but 480 is the VGA limit
    severity error;
	 
vga_clk <= vga_tick;

vga: vga_display generic map (clock_pixel_ratio => clock_pixel_ratio)
	  port map (clock, resetn, vga_tick, video_on_buf, hcount_buf, vcount_buf, HS,VS);
                 
iram: in_RAMgen generic map (nrows => npixels, ncols => npixels, FILE_IMG => FILE_IMG, INIT_VALUES => "YES")
--iram: in_RAMgen generic map (nrows => npixels, ncols => npixels, INIT_VALUES => "NO") -- this is if we don't initialize the memories
	   port map (clock, inRAM_idata, inRAM_add, inRAM_we, inRAM_en, inRAM_odata);
		
		-- Writing to this RAM is disabled here, but we can always enable to load new data
		inRAM_idata <= (others => '0'); inRAM_en <= '1'; inRAM_we <= '0';
		
		-- The following formulas only works if the image is square and the width size is a power of 2.
      inRAM_add <= vcount_buf(NBPR-1 downto 0) & hcount_buf(NBPR-1 downto 0);
		
		-- This is how we control what appears on the screen (only a square 2**NBPR x 2**NBPR)
		sel_RGB <= '1' when (hcount_buf < 2**NBPR) and (vcount_buf < 2**NBPR) else '0';
		
		with sel_RGB select
				in_RGB <= inRAM_odata(11 downto 0) when '1', -- only the 12 LSBs contain useful data
							 sw when others;		  

ro: my_rege generic map (N => nbits)
    port map (clock => clock, resetn => '1', E => vga_tick, sclr => '0', D => in_RGB, Q => RGB_buf);

    -- This is very important, zeros MUST BE WRITEN when we are in the back or front porch	 
	 RGB <= rgb_buf when video_on_buf = '1' else (others => '0');	 
	 -- Usually, the controller will perform this task by itself:
	 -- 'vga_display' provides hc, vc, so that we know that we are not in hc=[0,639], we MUST write zero

video_on <= video_on_buf;
hcount <= hcount_buf;
vcount <= vcount_buf;

end structure;
