---------------------------------------------------------------------------
-- This VHDL file was developed by Daniel Llamocca (2014).  It may be
-- freely copied and/or distributed at no cost.  Any persons using this
-- file for any purpose do so at their own risk, and are responsible for
-- the results of such use.  Daniel Llamocca does not guarantee that
-- this file is complete, correct, or fit for any particular purpose.
-- NO WARRANTY OF ANY KIND IS EXPRESSED OR IMPLIED.  This notice must
-- accompany any copy of this file.
--------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library UNISIM;
use UNISIM.vcomponents.all;

library work;
use work.pack_xtras.all;
--use work.lpm_components.all;

-- Nexys-4 Board:
-- 12-bit color: R (4 bits), G (4 bits), B (4 bits)
-- 4-bit grayscale images: R=G=B
entity vga_ctrl_12b is
	generic (TYPE_CTRL: string:= "MEMORY"; -- "12BASIC" (2048 colors), "BASIC" (8 colors), MEMORY
				clock_pixel_ratio : integer:= 4); -- (Available clock)/25MHz (pixel clock)
				                                  -- Only two available
	port ( clock: in std_logic;	
			 resetn: in std_logic; -- low level reset		   
			 SW: in std_logic_vector (11 downto 0);
			 R, G, B: out std_logic_vector (3 downto 0);
			 HS, VS: out std_logic);
			 --vga_clk: out std_logic;
			 -- debug signals
			 --video_on: out std_logic;
			 --hcount, vcount: out std_logic_vector (9 downto 0));
end vga_ctrl_12b;

architecture structure of vga_ctrl_12b is

	component vga_ctrl_simple
		generic (clock_pixel_ratio : integer:= 2;
					nbits: integer:= 3); -- number of bits for each pixel. Example: 3, 8, 12, 15 
		port ( clock: in std_logic;	
				 resetn: in std_logic;   
				 SW: in std_logic_vector (nbits-1 downto 0);
				 RGB: out std_logic_vector (nbits-1 downto 0);
				 HS, VS: out std_logic;
				 vga_clk: out std_logic;
				 -- debug signals
				 video_on: out std_logic;
				 hcount, vcount: out std_logic_vector (9 downto 0));
	end component;

	component vga_ctrl_ram
		generic (clock_pixel_ratio : integer:= 2;
					NPIXELS: INTEGER:= 256; -- Picture: NPIXELSxNPIXELS. Max for Nexys-4: 525x525
													-- So far, we only work with square images
					FILE_IMG: STRING:= "myimg.txt"; -- text file containing the image
					nbits: integer:= 12);  -- number of bits for each pixel. Example: 3, 8, 12, 15 
		port ( clock: in std_logic;	
				 resetn: in std_logic;	   
				 SW: in std_logic_vector (nbits-1 downto 0);
				 RGB: out std_logic_vector (nbits-1 downto 0);
				 HS, VS: out std_logic;
				 vga_clk: out std_logic;

				 -- debug signals
				 video_on: out std_logic;
				 hcount, vcount: out std_logic_vector (9 downto 0));
	end component;

   signal RGB: std_logic_vector (11 downto 0);
	
	signal vga_clk: std_logic;
	-- debug signals
	signal video_on: std_logic;
	signal hcount, vcount: std_logic_vector (9 downto 0);

begin

-- 12-BIT BACKGROUND: A single 12-bit color appears in the screen. SW11-SW0 control the color
-- Look for a table of RGB colors (only use the 4 MSBs for each color)
ga: if (TYPE_CTRL = "12BASIC") generate
		  basic12: vga_ctrl_simple generic map (clock_pixel_ratio => clock_pixel_ratio, nbits => 12)
					 port map (clock, resetn, SW, RGB, HS,VS, vga_clk, video_on, hcount, vcount);
			R <= RGB(11 downto 8); G <= RGB (7 downto 4); B <= RGB (3 downto 0);
	 end generate;

-- 3-BIT BACKGROUND: A single 3-bit color appears in the screen. SW2-SW0 control the color (only 8 color)
gb: if (TYPE_CTRL = "BASIC") generate
		  basic3: vga_ctrl_simple generic map (clock_pixel_ratio => clock_pixel_ratio, nbits => 3)
					 port map (clock, resetn, SW(2 downto 0), RGB(2 downto 0), HS,VS, vga_clk, video_on, hcount, vcount);
			R <= RGB(2)&RGB(2)&RGB(2)&RGB(2); G <= RGB(1)&RGB(1)&RGB(1)&RGB(1); B <= RGB(0)&RGB(0)&RGB(0)&RGB(0);
	 end generate;	 

-- tested for 16, 32, 64,
gbc: if (TYPE_CTRL = "MEMORY") generate
		  ramd: vga_ctrl_ram generic map (clock_pixel_ratio => clock_pixel_ratio, NPIXELS => 256, FILE_IMG => "myimg.txt", nbits => 12) -- 256x256
		  --ramd: vga_ctrl_ram generic map (clock_pixel_ratio => clock_pixel_ratio, NPIXELS => 128, FILE_IMG => "myimgt.txt", nbits => 12) -- 128x128
				  port map (clock, resetn, SW, RGB, HS,VS, vga_clk, video_on, hcount, vcount);
			R <= RGB(11 downto 8); G <= RGB (7 downto 4); B <= RGB (3 downto 0);
	 end generate;

end structure;
