function [x, y, z] = cordic_circular_esp (xin, yin, zin, mode, N)
% Special case, where the iteration 0 is repeated two times
%   xin, yin, zin: Initial values
%   mode = 0 --> Rotation
%   mode = 1 --> Vectoring
%   N: number of iterations (CORDIC equations: i = 0, 0, 0,1, ..., N-1):

% Iteration 'i' processes data x(i),y(i),z(i) and generates x(i+1),y(i+1),z(i+1)
% So, there will be iterations from i = 0 to i = N-1
%   and there will be data from x(0), y(0), z(0) to x(N), y(N), z(N)
%      There are N+2 iterations and N+3 indices
%      MATLAB does not take zero-valued indices, so in MATLAB we
%      go from iteration 1 (the zero) to iteration N+3.

%  x, y, z: vectors from 1 to N+3 (0,0,0,1,...,N-1,N in CORDIC notation)
    x(1) = xin; y(1) = yin; z(1) = zin; % Initial values
    
% CORDIC circular: i = 0, 0, 0,1,2, ..., N-1
% For now, let's fix the # of iterations to 16 and the FX format of the
    % angles (and inputs/outputs) to [16 14]
    
    % Scaling factor (depends on N)
        An = 1;    
        rng = [0 0 0:N-1];
        for j = rng
            An = An*sqrt(1 + 2^(-2*j)); % ==1.647*2 (and N --> inf)
        end
        F_Esc= 1/An; % Scaling factor
        disp ('Scaling Factor:'); disp (An);
        
    % Angles: theta(i) = atan(2^-i)
    
    % Angle Z: -pi to pi, == -3.14 to 3.14 rads. Thus we need 3 integer
    % bits. But with the angles we get, we realize that only 2 integer bits are required
    % The format [16 14] would work for this.
    % Input X and Y: The only restriction is the one given by the format.
    % Intermediate values of x,y,z: always plot them to find if values
    % overshoot (more than what it is allowed by the format)

    % Fixed-point representation: [16 14]
        q = quantizer('fixed', 'round', [16 14]);

    % Iterations: 'N' iterations from 0 to N-1. (1 to N in MATLAB)
    %    The 'N' iterations require data from 0 to N-1 (or 1 to N in MATLAB)
    %      x(0), y(0), z(0): initial values, x(N), y(N), z(N): results
    % MATLAB variable    CORDIC variable
    %           e(1)   == e(0)
    %           e(2)   == e(1)
    %              ........
    %           e(N-1) == e(N-2)
    %           e(N)   == e(N-1)
    
        % i=1 --> we start with 2^0, and angle 45
        for i = 1:N % compute angles for 0 to N-1 (in CORDIC notation)
            LUT_arg(i)= 2^-(i-1); % arguments
            e(i) = atan(LUT_arg(i)); % Angle for each arctan (radians)

            % Quantizying angles
            e_q(i) = quantize(q,e(i));

            % Note that by using [16 13], e_q(n=16) is 0. So, we might
            % wanna use more fractional bits

            alfa_hex = num2hex (q,e_q); % Angle in radians (hex)
            e_sexag(i) = (180/pi)*e(i); % Angle in 
        end


    %-- LUT: angles ei = 2^-i; i = 0 ... 15
    %-- i  Tan(alpha(i))	 alfa(i)      alfa(i)(radians)  hexadecimal [16 14]    integer value
    %-- 0   2^0  = 1 		  45				0.7854			3244					12868
    %-- 1	2^-1 = 0.5		  26.5650			0.4636			1dac					07596
    %-- 2	2^-2 = 0.25		  14.036243			0.2450			0fae					04014
    %-- 3	2^-3 = 0.125	  7.125016			0.1243			07f5					02037
    %-- 4	2^-4 = 0.0625	  3.576334			0.0624			03ff					01023
    %-- 5	2^-5 = 0.031250	  1.789910			0.0313			0200					00512
    %-- 6	2^-6 = 0.015625	  0.895174			0.0156			0100					00256
    %-- 7	2^-7 = 0.0078125  0.447614			0.0078			0080					00128

    Z_max = 0; % Variable that stores the maximum achievable angle
    % x,y,z: from index 1 to index N+3: 0,0,0,1,2,...,N
    
    for i = 1:N+2 % N iterations + 2 iterations (0,0, i = 1 ==> initial values
        if mode == 0
            % Rotation Mode
            if z(i) < 0
                d(i) = 1;
            else
                d(i) = -1;
            end
        else
            % Vectoring Mode
            if y(i) >= 0
                d(i) = 1;
            else
                d(i) = -1;
            end
        end
        % Iteration i operates on data i and generates data i+1
        % Here, for the index i (MATLAB), we actually are in the index
        % i - 1 in CORDIC
        if i <= 2 % First 2 iterations (0,0)
            Z_max = Z_max + e(1); % e(i) = e(i=1)
            x(i+1) = x(i) + (d(i)*y(i))*(2^-(1-1));
            y(i+1) = y(i) - (d(i)*x(i))*(2^-(1-1));
            z(i+1) = z(i) + (d(i)*e(1));
        else % Next N iterations (0,1,N-1)
            Z_max = Z_max + e(i-2);
            x(i+1) = x(i) + (d(i)*y(i))*(2^-(i-1-2));
            y(i+1) = y(i) - (d(i)*x(i))*(2^-(i-1-2));
            z(i+1) = z(i) + (d(i)*e(i-2));
        end
    end

    disp (alfa_hex);
    %t = 1:17; % this would be from 0 to 16 in cordic equations.
    %figure; plot (t,x,'r',t,y,'b',t,z,'k');

end